'use client';

import { useState, useEffect } from 'react';
import Image from 'next/image';
import Link from 'next/link';
import { Button } from '@/components/ui/button';
import { ChevronLeft, ChevronRight } from 'lucide-react';
import { Slider, SliderSlide } from '@/lib/data-store';

interface SliderComponentProps {
  slider: Slider;
  className?: string;
}

export default function SliderComponent({ slider, className = '' }: SliderComponentProps) {
  const [currentSlide, setCurrentSlide] = useState(0);
  const [isAutoPlaying, setIsAutoPlaying] = useState(false);

  const activeSlides = slider.slides.filter(slide => slide.isActive).sort((a, b) => a.order - b.order);

  // Auto-play functionality
  useEffect(() => {
    if (slider.settings.autoPlay && activeSlides.length > 1) {
      setIsAutoPlaying(true);
      const interval = setInterval(() => {
        setCurrentSlide(prev => 
          prev >= activeSlides.length - 1 ? 0 : prev + 1
        );
      }, slider.settings.autoPlayDelay);

      return () => clearInterval(interval);
    }
  }, [slider.settings.autoPlay, slider.settings.autoPlayDelay, activeSlides.length]);

  const nextSlide = () => {
    if (slider.settings.infinite || currentSlide < activeSlides.length - 1) {
      setCurrentSlide(prev => 
        prev >= activeSlides.length - 1 ? 0 : prev + 1
      );
    }
  };

  const prevSlide = () => {
    if (slider.settings.infinite || currentSlide > 0) {
      setCurrentSlide(prev => 
        prev <= 0 ? activeSlides.length - 1 : prev - 1
      );
    }
  };

  const goToSlide = (index: number) => {
    setCurrentSlide(index);
  };

  if (activeSlides.length === 0) {
    return null;
  }

  const currentSlideData = activeSlides[currentSlide];

  const getTextAlign = (position?: string) => {
    switch (position) {
      case 'left': return 'text-left';
      case 'right': return 'text-right';
      case 'center':
      default: return 'text-center';
    }
  };

  const getButtonVariant = (style?: string) => {
    switch (style) {
      case 'secondary': return 'secondary';
      case 'outline': return 'outline';
      case 'primary':
      default: return 'default';
    }
  };

  return (
    <div className={`relative w-full h-[500px] md:h-[600px] overflow-hidden ${className}`}>
      {/* Main Slide */}
      <div className="relative w-full h-full">
        {/* Background Image */}
        <div className="absolute inset-0">
          <Image
            src={currentSlideData.image}
            alt={currentSlideData.title}
            fill
            className="object-cover"
            priority={currentSlide === 0}
            sizes="100vw"
          />
          <div className="absolute inset-0 bg-black/40"></div>
        </div>

        {/* Content Overlay */}
        <div className="absolute inset-0 flex items-center justify-center">
          <div className="container mx-auto px-4">
            <div className={`${getTextAlign(currentSlideData.textPosition)} text-white max-w-4xl mx-auto`}>
              {currentSlideData.subtitle && (
                <p className="text-lg md:text-xl mb-4 text-blue-200">
                  {currentSlideData.subtitle}
                </p>
              )}
              <h1 className="text-3xl md:text-5xl lg:text-6xl font-bold mb-6">
                {currentSlideData.title}
              </h1>
              {currentSlideData.description && (
                <p className="text-lg md:text-xl mb-8 max-w-3xl mx-auto opacity-90">
                  {currentSlideData.description}
                </p>
              )}
              {currentSlideData.buttonText && currentSlideData.buttonLink && (
                <div className="flex gap-4 justify-center">
                  <Button 
                    size="lg" 
                    variant={getButtonVariant(currentSlideData.buttonStyle)}
                    className="px-8"
                    asChild
                  >
                    <Link href={currentSlideData.buttonLink}>
                      {currentSlideData.buttonText}
                    </Link>
                  </Button>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Navigation Arrows */}
      {slider.settings.showArrows && activeSlides.length > 1 && (
        <>
          <button
            onClick={prevSlide}
            className="absolute left-4 top-1/2 transform -translate-y-1/2 bg-white/20 hover:bg-white/30 text-white p-2 rounded-full transition-colors"
            aria-label="Previous slide"
          >
            <ChevronLeft className="h-6 w-6" />
          </button>
          <button
            onClick={nextSlide}
            className="absolute right-4 top-1/2 transform -translate-y-1/2 bg-white/20 hover:bg-white/30 text-white p-2 rounded-full transition-colors"
            aria-label="Next slide"
          >
            <ChevronRight className="h-6 w-6" />
          </button>
        </>
      )}

      {/* Dots Indicator */}
      {slider.settings.showDots && activeSlides.length > 1 && (
        <div className="absolute bottom-6 left-1/2 transform -translate-x-1/2 flex space-x-2">
          {activeSlides.map((_, index) => (
            <button
              key={index}
              onClick={() => goToSlide(index)}
              className={`w-3 h-3 rounded-full transition-colors ${
                index === currentSlide
                  ? 'bg-white'
                  : 'bg-white/50 hover:bg-white/75'
              }`}
              aria-label={`Go to slide ${index + 1}`}
            />
          ))}
        </div>
      )}
    </div>
  );
}